
  /*
   *  Object %name    : %
   *  State           :  %state%
   *  Creation date   :  Thu Jan 06 15:21:09 2005
   *  Last modified   :  %modify_time%
   */
  /** @file
   *  \brief A brief description of this module
   *
   *  \version CRYS_RSA_PSS_Sign.c#1:csrc:1
   *  \author ohads
   *  \remarks Copyright (C) 2004 by Discretix Technologies Ltd.
   *           All Rights reserved
   */



/************* Include Files ****************/
#include "DX_VOS_Mem.h"
#include "CRYS.h"
#include "CRYS_RSA_error.h"
#include "CRYS_RSA_Types.h"
#include "PLAT_SystemDep.h"
#include "dx_hw_defs.h"
#include "crys_host_op_code.h"

/************************ Defines ******************************/

/* canceling the lint warning:
   Use of goto is deprecated */
/*lint --e{801} */

/************************ Enums ******************************/


/************************ Typedefs ******************************/


/************************ Global Data ******************************/


/************* Private function prototype ****************/


/************************ Public Functions ******************************/

#ifndef _INTERNAL_CRYS_NO_RSA_SIGN_SUPPORT
/**********************************************************************************************************/
/**
   \brief RSA_SignInit initializes the Signing
   multi-call algorithm as defined in PKCS#1 v1.5 and 2.1

	The actual macro that will be used by the user is:
	CRYS_RSA_PSS_SignInit		 - for v2.1
	CRYS_RSA_PKCS1v15_SignInit	 - for v1.5

   This function does not do cryptographic processing. Rather, it
   prepares a context that is used by the Update
   and Finish functions.
   
   @param[in,out] UserContext_ptr - A pointer to a Context. The value
                                returned here must be passed to
                                the Update and Finish functions.
   @param[in] UserPrivKey_ptr - A pointer to the private key data
                            structure. \note The representation
                            (pair or quintuple) and hence the
                            algorithm (CRT or not) is determined
                            by the Private Key data
                            structure -using CRYS_Build_PrivKey
                            or CRYS_Build_PrivKeyCRT determines
                            which algorithm will be used.
   @param[in] hashFunc - The hash function to be used. Currently
                         only CRYS_PKCS1_SHA1 is supported. The
                         other hash functions recommended by PKCS#1
                         v2.1 are SHA-256/284/512.
   @param[in] MGF - The mask generation function. PKCS#1 v2.1
                    defines MGF1, so the only value allowed here is CRYS_PKCS1_MGF1.
   @param[in] SaltLen - The Length of the Salt buffer. relevant for PKCS#1 Ver 2.1 Only
   						Typical lengths are 0 and hLen (20 for SHA1)
   						The maximum length allowed is NSize - hLen - 2                    
   @param[in] PKCS1_ver - Ver 1.5 or 2.1, according to the functionality required
                       
   @return CRYSError_t - CRYS_OK, CRYS_BAD_PARAM
*/

CEXPORT_C CRYSError_t _DX_RSA_SignInit(CRYS_RSAPrivUserContext_t *UserContext_ptr,
                             CRYS_RSAUserPrivKey_t *UserPrivKey_ptr,
                             CRYS_RSA_HASH_OpMode_t hashFunc,
                             CRYS_PKCS1_MGF_t MGF,
                             DxUint16_t SaltLen,				  				  
                             CRYS_PKCS1_version PKCS1_ver)
{
 	return DX_HW_Dispatcher_ptr( DX_SEP_HOST_SEP_PROTOCOL_HOST_RSA_SIGN_INIT_OP_CODE, 6,
	   							 /* PARAMETERS: */
								 UserContext_ptr,
								 UserPrivKey_ptr,
								 hashFunc,
							 	 MGF,
								 SaltLen,				  				  
								 PKCS1_ver );

}/* _DX_RSA_SignInit */
				  
/**********************************************************************************************************/
/**
   \brief CRYS_RSA_PSS_SignUpdate processes the data to be signed
   in a given context.

   \note RSA_PSS_SignUpdate can be called multiple times
   with data 
   
	The actual macro that will be used by the users is (one of):
	CRYS_RSA_PSS_SignUpdate
	CRYS_RSA_PKCS1v15_SignUpdate

   \note CRYS_RSA_PSS_SignUpdate can be called multiple times
   with data 

   @param[in] UserContext_ptr - A pointer to a valid context,
                            as returned by CRYS_RSA_PSS_SignInit.
   @param[in] DataIn_ptr - A pointer to the data to sign.
   @param[in] DataInSize - The size, in bytes, of the data to sign.

   @return CRYSError_t - CRYS_OK, CRYS_BAD_PARAM
*/

CEXPORT_C CRYSError_t _DX_RSA_SignUpdate(CRYS_RSAPrivUserContext_t *UserContext_ptr,
                               DxUint8_t     *DataIn_ptr,
                               DxUint32_t     DataInSize)
{
 	return DX_HW_Dispatcher_ptr( DX_SEP_HOST_SEP_PROTOCOL_HOST_RSA_SIGN_UPDATE_OP_CODE, 3,
	   							 /* PARAMETERS: */
								 UserContext_ptr,
								 DataIn_ptr, DataInSize );

}/* _DX_RSA_SignUpdate */


/**********************************************************************************************************/
/**
   \brief _DX_RSA_SignFinish calculates the signature on the
   data passed to one or more calls to CRYS_RSA_PSS_SignUpdate,
   and releases the context.

	The actual macro that will be used by the user is:
	CRYS_RSA_PSS_SignFinish			- for v2.1
	CRYS_RSA_PKCS1v15_SignFinish	- for v1.5
	
   @param[in,out] UserContext_ptr - A pointer to the Context
                                initialized by the SignInit function 
                                and used by the SignUpdate function
   @param[out] Output_ptr - A pointer to the signature. 
                            The buffer must be at least PrivKey_ptr->N.len bytes long 
                            (that is, the size of the modulus, in bytes).
   @param[in,out] OutputSize_ptr - A pointer to the Signature Size value - the input value
                                   is the signature buffer size allocated, the output value is 
                                   the signature size used.
                                   The buffer must be at least PrivKey_ptr->N.len bytes long 
                                   (that is, the size of the modulus, in bytes).

   @return CRYSError_t - CRYS_OK,
                         CRYS_RSA_INVALID_USER_CONTEXT_POINTER_ERROR,
                         CRYS_RSA_USER_CONTEXT_VALIDATION_TAG_ERROR,
                         CRYS_RSA_INVALID_OUTPUT_POINTER_ERROR,
                         CRYS_RSA_INVALID_SIGNATURE_BUFFER_SIZE,
                         CRYS_RSA_HASH_ILLEGAL_OPERATION_MODE_ERROR,
                         CRYS_RSA_PKCS1_VER_ARG_ERROR
*/

CEXPORT_C CRYSError_t _DX_RSA_SignFinish(CRYS_RSAPrivUserContext_t *UserContext_ptr,
                               DxUint8_t                   *Output_ptr,
                               DxUint16_t                  *OutputSize_ptr)  
{
 	return DX_HW_Dispatcher_ptr( DX_SEP_HOST_SEP_PROTOCOL_HOST_RSA_SIGN_FINISH_OP_CODE, 3,
	   							 /* PARAMETERS: */
								 UserContext_ptr,
								 Output_ptr, OutputSize_ptr );

}/* _DX_RSA_SignFinish */
				    

/**********************************************************************************************************/
/**
   @brief
   RSA_Sign implements the Signing algorithm as defined
   in PKCS#1 v1.5 or v2.1, using a single function

   The actual macros that will be used by the user are:
   CRYS_RSA_PKCS1v15_Sign		- for v1.5
   CRYS_RSA_PKCS1v15_SHA1_Sign	- for v1.5 data is hashed with SHA1
   CRYS_RSA_PKCS1v15_MD5_Sign	- for v1.5 data is hashed with MD5
   CRYS_RSA_PSS_Sign			- for v2.1
   CRYS_RSA_PSS_SHA1_Sign		- for v2.1 data is hashed with SHA1
					- for v2.1 MD5 is not supported, since 
					according to the PKCS#1 ver2.1 it is not recommended
   
   @param[in] UserContext_ptr - A pointer to a Context. For the use of the 
   								function as a space to work on.
   @param[in] UserPrivKey_ptr - A pointer to the private key data
                            structure of the user. \note The representation
                            (pair or quintuple) and hence the
                            algorithm (CRT or not) is determined
                            by the Private Key data
                            structure - using CRYS_Build_PrivKey
                            or CRYS_Build_PrivKeyCRT determines
                            which algorithm will be used.
   @param[in] hashFunc - The hash function to be used. Currently
                         only CRYS_PKCS1_SHA1 and CRYS_PKCS1_MD5
                         are supported. 
   @param[in] MGF - The mask generation function. Only for PKCS#1 v2.1
                    defines MGF1, so the only value allowed for v2.1 
                    is CRYS_PKCS1_MGF1. 
   @param[in] SaltLen - The Length of the Salt buffer. relevant for PKCS#1 Ver 2.1 Only
   						Typical lengths are 0 and hLen (20 for SHA1)
   						The maximum length allowed is NSize - hLen - 2                                                                                       
   @param[in] DataIn_ptr - A pointer to the data to sign.
   @param[in] DataInSize - The size, in bytes, of the data to sign.
   @param[out] Output_ptr - A pointer to the signature. 
                            The buffer must be at least PrivKey_ptr->N.len bytes long 
                            (that is, the size of the modulus in bytes).
   @param[in,out] OutputSize_ptr - A pointer to the Signature Size value - the input value
                            is the signature buffer size allocated, the output value is 
                            the signature size actually used.
                            The buffer must be at least PrivKey_ptr->N.len bytes long 
                            (that is, the size of the modulus in bytes).
   @param[in] PKCS1_ver - Ver 1.5 or 2.1, according to the functionality required
      
   @return CRYSError_t - CRYS_OK,
                         CRYS_RSA_INVALID_USER_CONTEXT_POINTER_ERROR,
                         CRYS_RSA_INVALID_PRIV_KEY_STRUCT_POINTER_ERROR,
                         CRYS_RSA_PRIV_KEY_VALIDATION_TAG_ERROR,
                         CRYS_RSA_HASH_ILLEGAL_OPERATION_MODE_ERROR,
                         CRYS_RSA_MGF_ILLEGAL_ARG_ERROR,
                         CRYS_RSA_PKCS1_VER_ARG_ERROR,
                         CRYS_RSA_INVALID_MESSAGE_DATA_SIZE,
                         CRYS_RSA_INVALID_OUTPUT_POINTER_ERROR,
                         CRYS_RSA_INVALID_SIGNATURE_BUFFER_SIZE
*/


CEXPORT_C CRYSError_t _DX_RSA_Sign(CRYS_RSAPrivUserContext_t *UserContext_ptr,
                         CRYS_RSAUserPrivKey_t *UserPrivKey_ptr,
                         CRYS_RSA_HASH_OpMode_t hashFunc,
                         CRYS_PKCS1_MGF_t MGF,
                         DxUint16_t SaltLen,				  				  			      
                         DxUint8_t     *DataIn_ptr,
                         DxUint32_t     DataInSize,
                         DxUint8_t     *Output_ptr,
                         DxUint16_t    *OutputSize_ptr,
                         CRYS_PKCS1_version PKCS1_ver)

{
 	return DX_HW_Dispatcher_ptr( DX_SEP_HOST_SEP_PROTOCOL_HOST_RSA_SIGN_OP_CODE, 10,
	   							 /* PARAMETERS: */
								 UserContext_ptr,
								 UserPrivKey_ptr,
								 hashFunc,
								 MGF,
								 SaltLen,				  				  			      
								 DataIn_ptr,
								 DataInSize,
								 Output_ptr,
								 OutputSize_ptr,
								 PKCS1_ver );

}/* END OF _DX_RSA_Sign */

#endif /*_INTERNAL_CRYS_NO_RSA_SIGN_SUPPORT*/
